# Copyright (c) 2015, 2016, Oracle and/or its affiliates. All rights reserved.
# 
# JRF MT WLST Command Line Interface - to provision partition for upper stack components, e.g. OPSS, OWSM, etc.
#
# This script contains commands to:
# 1. get the configurable attributes, required, to provision a partition, for a given component type;
# 2. provision a partition for a given component type, using the properties file;
#
# Use debug() to trun printing debug messages on/off
# Use dumpStack() when seeing error, to dump the entire current stacktrace on console
# ---------------------------------------------------------------------------------------------

from com.oracle.weblogic.lifecycle.provisioning.wlst import ProvisioningWLSTCLI

global objProvisioningWLSTCLI

msgFormatter = WLS_ON.getWLSTMsgFormatter()

# ---------------------------------------------------------------------------------------------
# Uses the restURI, username, and password, to authenticate and talk to the REST endpoint.
#
# param: restURI - REST base URI, used to construct the REST endpoint.
# param: username - authentication credentials to authenticate to the REST endpoint.
# param: password - authentication credentials to authenticate to the REST endpoint.
# ---------------------------------------------------------------------------------------------
def connectToREST(restURI, username, password):
    try:
        print 'Connecting to REST...'
        global objProvisioningWLSTCLI
        objProvisioningWLSTCLI = ProvisioningWLSTCLI(restURI, username, password, WLS_ON)
        print 'Ok'
    except Exception,e:
        print 'Error while connecting to REST.', e.getMessage()
        WLS_ON.setDumpStackThrowable(e)
        raiseWLSTException(e)

# ---------------------------------------------------------------------------------------------
# get all the configurable properties for the given component type, and for
# the dependent components, required to provision a partition.
#
# param: componentType - component type for which configurable properties are to be retrieved
# param: configFilename - output filename to which to write all the configurable properties
# ---------------------------------------------------------------------------------------------
def getPartitionProvisioningConfigAttrs(componentType, configFilename):
    try:
        print 'Getting partition provisioning configurable attributes for component \"%s\"' % componentType
        try:
            global objProvisioningWLSTCLI
            objProvisioningWLSTCLI.getPartitionProvisioningConfigAttrs(componentType, configFilename)
        except NameError:
            raise Exception, "Not connected. Please use connectToREST(restURI, username, password) to get connected."
        print 'Done'
    except Exception,e:
        print 'Error while trying to get partition provisioning configurable attributes for component \"%s\". Error: %s' % (componentType, e.getMessage())
        WLS_ON.setDumpStackThrowable(e)
        raiseWLSTException(e)

# ---------------------------------------------------------------------------------------------
# provision partition for the given component type, using the properties supplied
#
# param: partitionName - partition name
# param: componentType - component type to provision
# param: configFilename - input properties file containing all the configurable properties
# ---------------------------------------------------------------------------------------------
def provisionPartition(partitionName, componentType, configFilename):
    try:
        print 'Provisioning partition \"%s\", for component \"%s\"' % (partitionName, componentType)
        try:
            global objProvisioningWLSTCLI
            objProvisioningWLSTCLI.provisionPartition(partitionName, componentType, configFilename)
        except NameError:
            raise Exception, "Not connected. Please use connectToREST(restURI, username, password) to get connected."
        print 'Done'
    except Exception,e:
        print 'Error while trying to provision partition \"%s\", for component \"%s\". Error: %s' % (partitionName, componentType, e.getMessage())
        WLS_ON.setDumpStackThrowable(e)
        raiseWLSTException(e)

# ---------------------------------------------------------------------------------------------
# Deprovision partition for the given component type
#
# param: partitionName - partition name
# param: componentType - component type to provision
# ---------------------------------------------------------------------------------------------
def deprovisionPartition(partitionName, componentType, configFileName=None):
    try:
        print msgFormatter.getDeprovisionPartitionStartMsg(partitionName, componentType)
        try:
            global objProvisioningWLSTCLI
            objProvisioningWLSTCLI.deprovisionPartition(partitionName, componentType, configFileName)
        except NameError:
            raise Exception, "Not connected. Please use connectToREST(restURI, username, password) to get connected."
        print msgFormatter.getDeprovisionPartitionCompletedMsg(partitionName, componentType)
    except Exception,e:
        print msgFormatter.getDeprovisionPartitionErrorMsg(partitionName, componentType, e.getMessage())
        WLS_ON.setDumpStackThrowable(e)
        raiseWLSTException(e)
